# Logo Cleanup Implementation Guide

## What Was Added

Added `GL.removeLogoByAbbr(abbr)` function to `gl_common.jsxinc` that:
- Removes logo footage items from the AE project by team abbreviation
- Runs **after each team renders** (inside the loop, not at the end)
- Matches files like `PHI.png`, `BUF.jpg`, etc.
- Keeps project clean throughout batch processing

---

## How to Use

### In Your JSX Scripts

Add the cleanup call **at the end of each team's render loop**, after `GL.rqRenderTo()`:

```javascript
// --- per team ---
for (var i=0; i<todo.length; i++){
  var t = todo[i];

  // 1. Apply colors
  var smart = GL.computeSmartColors(GL.safeColor(t.primary), GL.safeColor(t.secondary));

  // 2. Replace logo
  GL.replaceLogo(comp, LOGO_LAYER, t.league, t.abbr, logoOpts);

  // 3. Apply other styling...

  // 4. Render
  if (!NO_RENDER){
    var lc = GL.leagueAndConfForPath(t.league, t.conference);
    var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
    GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
  }

  // 5. CLEANUP - Remove logo from project (NEW!)
  var removed = GL.removeLogoByAbbr(t.abbr);
  if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
}
```

---

## Files Already Updated

✅ `batch_ScoreAnimations1-TD.jsx` - Has cleanup added

---

## Files To Update

Add the cleanup call to these GL-based files:

- [ ] `batch_ScoreAnimation2-FG-logospin.jsx`
- [ ] `batch_ScoreAnimation2-FG-logospin-CFB.jsx`
- [ ] `batch_ScoreAnimations2-TD.jsx`
- [ ] `batch_ScoreAnimations2-TD-CFB.jsx`
- [ ] `batch_ScoreAnimations1-FG.jsx`
- [ ] `batch_HomeRun_3.jsx`
- [ ] `batch_RunScored_3.jsx`
- [ ] `batch_Homerun_H123.jsx`
- [ ] `batch_RunScored_H123.jsx`
- [ ] `batch_GOAL_Animation_1.jsx`

---

## Manual Update Steps

For each file above:

1. **Find the render block** - Search for `GL.rqRenderTo`
2. **Add cleanup after render** - After the `GL.rqRenderTo()` call (but still inside the loop)
3. **Add these 3 lines:**
   ```javascript
   // Clean up logo from project after render
   var removed = GL.removeLogoByAbbr(t.abbr);
   if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
   ```

---

## Example: Before & After

### BEFORE:
```javascript
for (var i=0; i<todo.length; i++){
  var t = todo[i];

  GL.replaceLogo(comp, LOGO_LAYER, t.league, t.abbr, logoOpts);

  if (!NO_RENDER){
    var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
    GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
  }
  // Next team starts here with OLD logo still in project
}
```

### AFTER:
```javascript
for (var i=0; i<todo.length; i++){
  var t = todo[i];

  GL.replaceLogo(comp, LOGO_LAYER, t.league, t.abbr, logoOpts);

  if (!NO_RENDER){
    var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
    GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
  }

  // NEW: Remove logo immediately after render
  var removed = GL.removeLogoByAbbr(t.abbr);
  if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
  // Next team starts with clean project
}
```

---

## What Gets Removed

The function removes FootageItems matching:
- `{ABBR}.png`
- `{ABBR}.jpg`
- `{ABBR}.jpeg`
- `{ABBR}.svg`
- `{ABBR}.ai`
- `{ABBR}.psd`

Where `{ABBR}` is the team abbreviation (case-insensitive).

**Examples:**
- Team `PHI` → Removes `PHI.png`, `phi.PNG`, etc.
- Team `BUF` → Removes `BUF.jpg`, `buf.JPG`, etc.

---

## Safety Features

1. **Only removes imported logos** - Doesn't affect other PNG assets
2. **Matches exact filenames** - Won't remove `PHI_old.png` or `PHI-2024.png`
3. **Safe to call multiple times** - If logo already removed, returns 0
4. **Doesn't delete files from disk** - Only removes from AE project

---

## Testing

Test with a small subset first:

```python
# In your Python runner
RUN_MODE = "subset"
SUBSET = ["PHI", "BUF"]  # Just 2 teams to test
```

Watch After Effects project panel - you should see:
1. PHI logo imported
2. PHI rendered
3. PHI logo removed from project
4. BUF logo imported (project only has 1 logo at a time)
5. BUF rendered
6. BUF logo removed

---

## Optional: Disable Cleanup

If you want to keep logos in the project for debugging:

**Option 1: Comment out the cleanup**
```javascript
// var removed = GL.removeLogoByAbbr(t.abbr);
// if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
```

**Option 2: Add environment variable control**
```python
# In Python runner
env["AE_CLEANUP_LOGOS"] = "0"  # Set to "1" to enable cleanup
```

```javascript
// In JSX (add this check)
var CLEANUP = (GL.env("AE_CLEANUP_LOGOS","1") === "1");
if (CLEANUP){
  var removed = GL.removeLogoByAbbr(t.abbr);
  if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
}
```

---

**Ready to use!** The function is in `gl_common.jsxinc` and one example file has been updated.
